(function(wp){
    const { registerPlugin } = wp.plugins;
    const { PluginSidebar } = wp.editPost;
    const { PanelBody, TextControl, Button, Notice } = wp.components;
    const { useSelect, useDispatch } = wp.data;
    const { Fragment, useState, useEffect, useCallback, useMemo } = wp.element;

    // Helpers (keyword extraction, heading count, link parsing etc.)
    function extractKeywords(content) {
        const stopwords = ['the','is','at','which','on','and','a','to','of','in','it','with','as','for','its','by','an','but','be','are','this','that'];
        let words = content.replace(/(<([^>]+)>)/gi, "").replace(/[^\w\s]/g, "").toLowerCase().split(/\s+/);
        let freq = {};
        words.forEach(w => {
            if (!stopwords.includes(w) && w.length > 2) {
                freq[w] = (freq[w] || 0) + 1;
            }
        });
        return Object.entries(freq).sort((a,b) => b[1]-a[1]).slice(0,8).map(x => x[0]);
    }
    function countHeadings(content) {
        return (content.match(/<h[2-6][^>]*>/g) || []).length;
    }
    function getInternalLinks(content, siteUrl) {
        const regex = /<a [^>]*href=["']([^"']+)["']/g;
        let count = 0, match;
        while ((match = regex.exec(content))) {
            if (match[1].startsWith(siteUrl)) count++;
        }
        return count;
    }
    function countImagesWithAlt(content) {
        const imgTags = content.match(/<img [^>]*>/g) || [];
        return imgTags.filter(tag => /alt=["'][^"']+["']/i.test(tag)).length;
    }
    function getLinks(content) {
        const regex = /<a\s[^>]*?href=["']([^"']+)["']/gi;
        let links = [], match;
        while ((match = regex.exec(content))) {
            links.push(match[1]);
        }
        return links;
    }
    function classifyLinks(links, siteUrl) {
        let internal = 0, external = 0, broken = 0;
        links.forEach(link => {
            if (!link || link.startsWith('#')) return;
            try {
                const url = new URL(link, siteUrl);
                if(url.origin === siteUrl) internal++;
                else external++;
            } catch {
                broken++;
            }
        });
        return { internal, external, broken };
    }
    function keywordDensity(text, keyword) {
        if (!keyword) return 0;
        const words = text.toLowerCase().split(/\s+/);
        const count = words.filter(w => w === keyword.toLowerCase()).length;
        return words.length ? (count / words.length) * 100 : 0;
    }
    function averageSentenceLength(text) {
        const sentences = text.match(/[^.!?]+[.!?]+/g) || [];
        if (!sentences.length) return 0;
        const words = text.trim().split(/\s+/).length;
        return words / sentences.length;
    }
    function passiveVoicePercent(text) {
        const sentences = text.match(/[^.!?]+[.!?]+/g) || [];
        if (!sentences.length) return 0;
        const passiveRegex = /\b(was|were|is|are|been|be|being)\b\s+\w+ed\b/;
        let passiveCount = sentences.reduce((acc, s) => acc + (passiveRegex.test(s.toLowerCase()) ? 1 : 0), 0);
        return (passiveCount / sentences.length) * 100;
    }
    function readabilitySuggestions(avgSentenceLen, passivePercent) {
        const suggestions = [];
        if (avgSentenceLen > 20) suggestions.push('Try shorter sentences');
        if (passivePercent > 15) suggestions.push('Reduce passive voice');
        return suggestions;
    }

    const Sidebar = () => {
        const { meta, postContent, postTitle } = useSelect(select => {
            const editor = select('core/editor');
            return {
                meta: editor ? editor.getEditedPostAttribute('meta') || {} : {},
                postContent: editor ? editor.getEditedPostContent() || '' : '',
                postTitle: editor ? editor.getEditedPostAttribute('title') || '' : ''
            };
        }, []);

        const { editPost } = useDispatch('core/editor');
        const fallbackImageUrl = 'https://fajaede.nl/wp-content/uploads/fallback-og-image.png';
        const featuredImageUrl = meta.featured_image_url || fallbackImageUrl;
        const currentKeyword = meta._fajaedeseo_focus_keyword || '';
        const [aiLoading, setAiLoading] = useState(false);
        const [seoScore, setSeoScore] = useState(meta._fajaedeseo_seo_score || 0);
        const [feedback, setFeedback] = useState([]);
        const canonical = meta._fajaedeseo_canonical || window.location.origin;
        const [canonicalUrl, setCanonicalUrl] = useState(canonical);
        const siteUrl = window.location.origin;

        const keywordSuggestions = useMemo(() => extractKeywords(postContent), [postContent]);
        const headingCount = useMemo(() => countHeadings(postContent), [postContent]);
        const internalLinkCount = useMemo(() => getInternalLinks(postContent, siteUrl), [postContent, siteUrl]);
        const imagesWithAltCount = useMemo(() => countImagesWithAlt(postContent), [postContent]);
        const links = useMemo(() => getLinks(postContent), [postContent]);
        const linkStats = useMemo(() => classifyLinks(links, siteUrl), [links, siteUrl]);
        const analysisData = useMemo(() => ({
            density: keywordDensity(postContent, currentKeyword),
            avgSentLen: averageSentenceLength(postContent),
            passivePercent: passiveVoicePercent(postContent)
        }), [postContent, currentKeyword]);
        const writingSuggestions = useMemo(() =>
            readabilitySuggestions(analysisData.avgSentLen, analysisData.passivePercent),
            [analysisData.avgSentLen, analysisData.passivePercent]
        );

        const calculateScore = useCallback(() => {
            let score = 0, localFeedback = [];
            if (meta._fajaedeseo_meta_title) {
                const len = meta._fajaedeseo_meta_title.length;
                if (len >= 40 && len <= 60) score++;
                else localFeedback.push('Title length (40–60 chars)');
                if (currentKeyword && meta._fajaedeseo_meta_title.toLowerCase().includes(currentKeyword.toLowerCase())) score++;
                else if (currentKeyword) localFeedback.push('Keyword in title');
            } else { localFeedback.push('Add a meta title'); }
            if (meta._fajaedeseo_meta_desc) {
                const len = meta._fajaedeseo_meta_desc.length;
                if (len > 80 && len <= 160) score++;
                else localFeedback.push('Description length (80–160 chars)');
                if (currentKeyword && meta._fajaedeseo_meta_desc.toLowerCase().includes(currentKeyword.toLowerCase())) score++;
                else if (currentKeyword) localFeedback.push('Keyword in description');
            } else { localFeedback.push('Add a meta description'); }
            if (headingCount >= 2) score++; else localFeedback.push('Add more headings');
            if (imagesWithAltCount > 0) score++; else localFeedback.push('Add alt text to images');
            return { score, max: 6, localFeedback };
        }, [meta._fajaedeseo_meta_title, meta._fajaedeseo_meta_desc, currentKeyword, headingCount, imagesWithAltCount]);

        useEffect(() => {
            const { score, localFeedback } = calculateScore();
            setSeoScore(score);
            setFeedback(localFeedback);
            if (parseInt(meta._fajaedeseo_seo_score) !== score) {
                editPost({ meta: { ...meta, _fajaedeseo_seo_score: score } });
            }
        }, [meta, calculateScore, editPost]);

        const setMeta = useCallback((key, value) => {
            if (value === undefined || value === null) return;
            editPost({ meta: { ...meta, [key]: value } });
        }, [meta, editPost]);

        const updateKeyword = useCallback((newKeyword) => {
            setMeta('_fajaedeseo_focus_keyword', newKeyword);
        }, [setMeta]);

        const aiSuggest = useCallback((field) => {
            setAiLoading(true);
            let suggestion;
            if (field === '_fajaedeseo_meta_title') {
                suggestion = `Top ${postTitle || 'your keyword'} | ${document.title.replace(' ‹', '').split('—')[0].trim()}`;
            } else {
                suggestion = `Discover ${postTitle || 'your topic'} on ${document.title.replace(' ‹', '').split('—')[0].trim()}. Uncover insights, tips, and value in every read!`;
            }
            setTimeout(() => {
                setMeta(field, suggestion);
                setAiLoading(false);
            }, 700);
        }, [postTitle, setMeta]);

        const scoreColor = useCallback((val) => val >= 5 ? 'green' : val >= 3 ? 'orange' : 'red', []);

        const saveCanonical = useCallback(() => {
            setMeta('_fajaedeseo_canonical', canonicalUrl);
        }, [canonicalUrl, setMeta]);

        return (
            wp.element.createElement(Fragment, null,
                wp.element.createElement(PluginSidebar, { name: 'fajaedeseo-sidebar', title: 'fajaedeSEO', icon: 'admin-site-alt3' },
                    wp.element.createElement(PanelBody, { title: 'SEO Meta Fields', initialOpen: true },
                        wp.element.createElement(TextControl, {
                            label: 'Focus Keyword',
                            value: currentKeyword,
                            onChange: updateKeyword,
                            help: 'Target phrase for score and suggestions.'
                        }),
                        wp.element.createElement('div', { style: { marginBottom: '10px' } },
                            wp.element.createElement('strong', null, 'Keyword Suggestions: '),
                            keywordSuggestions.map(kw =>
                                wp.element.createElement('span', {
                                    key: kw,
                                    style: {
                                        background: kw === currentKeyword ? '#0073aa' : '#efefef',
                                        color: kw === currentKeyword ? '#fff' : '#000',
                                        borderRadius: '12px',
                                        margin: '2px',
                                        padding: '4px 9px',
                                        fontSize: '13px',
                                        cursor: 'pointer',
                                        display: 'inline-block'
                                    },
                                    onClick: () => updateKeyword(kw)
                                }, kw)
                            )
                        ),
                        wp.element.createElement(TextControl, {
                            label: 'AI Meta Title',
                            value: meta._fajaedeseo_meta_title || '',
                            onChange: (val) => setMeta('_fajaedeseo_meta_title', val),
                            help: '40–60 chars, use keyword.'
                        }),
                        wp.element.createElement(Button, {
                            isSmall: true,
                            isSecondary: true,
                            disabled: aiLoading,
                            style: { marginBottom: '12px' },
                            onClick: () => aiSuggest('_fajaedeseo_meta_title')
                        }, aiLoading ? '…' : '🔮AI Title Suggest'),
                        wp.element.createElement(TextControl, {
                            label: 'AI Meta Description',
                            value: meta._fajaedeseo_meta_desc || '',
                            onChange: (val) => setMeta('_fajaedeseo_meta_desc', val),
                            help: '80–160 chars, summarize content.'
                        }),
                        wp.element.createElement(Button, {
                            isSmall: true,
                            isSecondary: true,
                            disabled: aiLoading,
                            onClick: () => aiSuggest('_fajaedeseo_meta_desc')
                        }, aiLoading ? '…' : '🔮AI Desc Suggest'),
                        wp.element.createElement('div', { style: { marginTop: '16px', textAlign: 'center' } },
                            wp.element.createElement('p', { style: { marginBottom: '6px', fontWeight: '600' } }, 'Social Preview Image'),
                            wp.element.createElement('img', {
                                src: featuredImageUrl,
                                alt: 'Social Share Preview Image',
                                style: { maxWidth: '100%', height: 'auto', borderRadius: '8px', border: '1px solid #ddd' }
                            })
                        ),
                        wp.element.createElement('div', { style: { marginTop: '10px', marginBottom: '10px' } },
                            wp.element.createElement('strong', {
                                style: {
                                    padding: '6px 14px',
                                    borderRadius: '18px',
                                    background: scoreColor(seoScore),
                                    color: '#fff',
                                    display: 'inline-block',
                                    fontWeight: 'bold'
                                }
                            }, `SEO Score: ${seoScore} / 6`)
                        ),
                        feedback.length
                            ? wp.element.createElement(Notice, { status: 'warning', isDismissible: false }, 'Suggestions: ', feedback.join(', '))
                            : wp.element.createElement(Notice, { status: 'success', isDismissible: false }, 'Great job! Keep it up.')
                    ),
                    wp.element.createElement(PanelBody, { title: 'SEO Writing Analysis', initialOpen: false },
                        wp.element.createElement('div', null,
                            'Keyword Density: ',
                            wp.element.createElement('span', { style: { fontWeight: '600', color: analysisData.density >= 0.5 ? 'green' : 'red' } }, `${analysisData.density.toFixed(2)}%`)
                        ),
                        wp.element.createElement('div', null,
                            'Average Sentence Length: ',
                            wp.element.createElement('span', { style: { fontWeight: '600', color: analysisData.avgSentLen <= 20 ? 'green' : 'red' } }, analysisData.avgSentLen.toFixed(1))
                        ),
                        wp.element.createElement('div', null,
                            'Passive Voice Usage: ',
                            wp.element.createElement('span', { style: { fontWeight: '600', color: analysisData.passivePercent <= 15 ? 'green' : 'red' } }, `${analysisData.passivePercent.toFixed(1)}%`)
                        ),
                        writingSuggestions.length
                            ? wp.element.createElement(Notice, { status: 'warning', isDismissible: false }, 'Writing Tips: ', writingSuggestions.join(', '))
                            : wp.element.createElement(Notice, { status: 'success', isDismissible: false }, 'Good writing style detected.')
                    ),
                    wp.element.createElement(PanelBody, { title: 'Content Analysis', initialOpen: false },
                        wp.element.createElement('div', null,
                            'Headings found: ',
                            wp.element.createElement('span', { style: { color: headingCount >= 2 ? 'green' : 'red', fontWeight: '600' } }, headingCount)
                        ),
                        wp.element.createElement('div', null,
                            'Internal links: ',
                            wp.element.createElement('span', { style: { color: internalLinkCount > 0 ? 'green' : 'red', fontWeight: '600' } }, internalLinkCount)
                        ),
                        wp.element.createElement('div', null,
                            'Images with alt text: ',
                            wp.element.createElement('span', { style: { color: imagesWithAltCount > 0 ? 'green' : 'red', fontWeight: '600' } }, imagesWithAltCount)
                        )
                    ),
                    wp.element.createElement(PanelBody, { title: 'Links Analyse', initialOpen: false },
                        wp.element.createElement('div', null, `Internal links: ${linkStats.internal}`),
                        wp.element.createElement('div', null, `External links: ${linkStats.external}`),
                        wp.element.createElement('div', null, `Broken links: ${linkStats.broken}`),
                        linkStats.broken ?
                            wp.element.createElement(Notice, { status: 'error', isDismissible: false }, 'Let op: Er zijn gebroken links!')
                            : null
                    ),
                    wp.element.createElement(PanelBody, { title: 'Canonical URL', initialOpen: false },
                        wp.element.createElement(TextControl, {
                            label: 'Canonical URL',
                            value: canonicalUrl,
                            onChange: (val) => setCanonicalUrl(val),
                            help: 'Dit is de canonical tag die aan deze pagina wordt toegevoegd. Pas aan indien nodig.'
                        }),
                        wp.element.createElement(Button, {
                            isPrimary: true,
                            style: { marginTop: '10px' },
                            onClick: saveCanonical
                        }, 'Opslaan')
                    ),
                    wp.element.createElement('div', { style: { marginTop: '28px', color:'#389149', fontWeight:'bold'} },
                        '— Alles onder fajaedeSEO plugin —'
                    )
                )
            )
        );
    };

    registerPlugin('fajaedeseo', {
        render: Sidebar,
        icon: 'admin-site-alt3'
    });

})(window.wp);
