<?php
/*
Plugin Name: FajaedeSEO AI
Plugin URI: https://www.fajaede.nl/donate
Description: SEO plugin voor WordPress met dashboard, focus keywords, meta tags, analyse, XML Sitemap, robots.txt beheer en uitgebreide SEO Gutenberg sidebar.
Version: 1.9
Author: fEseo 
Author URI: https://www.fajaede.nl
License: GPL2
Text Domain: fajaedeseo
*/

defined('ABSPATH') or exit;

define('FAJAEDESEO_OG_IMAGE_FALLBACK', esc_url(plugins_url('assets/fallback-og-image.jpg', __FILE__)));

// ================= Register Meta en Settings =================
function fajaedeseo_register_meta_and_settings() {
    $post_types = ['post', 'page'];
    foreach ($post_types as $post_type) {
        register_post_meta($post_type, '_fajaedeseo_meta_title', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'auth_callback' => fn() => current_user_can('edit_posts'),
        ]);
        register_post_meta($post_type, '_fajaedeseo_meta_desc', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
            'auth_callback' => fn() => current_user_can('edit_posts'),
        ]);
        register_post_meta($post_type, '_fajaedeseo_focus_keyword', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'auth_callback' => fn() => current_user_can('edit_posts'),
        ]);
        register_post_meta($post_type, '_fajaedeseo_seo_score', [
            'show_in_rest' => true,
            'single' => true,
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'auth_callback' => fn() => current_user_can('edit_posts'),
        ]);
    }

    // Homepagina SEO opties
    register_setting('fajaedeseo_home_settings_group', 'fajaedeseo_home_seo_title', [
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('fajaedeseo_home_settings_group', 'fajaedeseo_home_seo_desc', [
        'sanitize_callback' => 'sanitize_textarea_field',
    ]);

    // Algemene plugin instellingen
    register_setting('fajaede_seo_settings_group', 'fajaede_robots_txt', [
        'sanitize_callback' => 'sanitize_textarea_field'
    ]);
    register_setting('fajaede_seo_settings_group', 'fajaede_twitter_handle', [
        'sanitize_callback' => 'sanitize_text_field'
    ]);
    register_setting('fajaede_seo_settings_group', 'fajaede_facebook_handle', [
        'sanitize_callback' => 'sanitize_text_field'
    ]);
}
add_action('init', 'fajaedeseo_register_meta_and_settings');



// ================ Admin menu + Settings pagina =================
function fajaedeseo_render_settings_page() {
    $default_robots_txt = "User-agent: *\nDisallow: /wp-admin/\nAllow: /wp-admin/admin-ajax.php\n\nSitemap: " . esc_url(home_url('/fajaede_sitemap/'));
    $robots_txt = get_option('fajaede_robots_txt', $default_robots_txt);
    $twitter_handle = get_option('fajaede_twitter_handle', '');
    $facebook_handle = get_option('fajaede_facebook_handle', '');
    $home_title = get_option('fajaedeseo_home_seo_title', get_bloginfo('name'));
    $home_desc = get_option('fajaedeseo_home_seo_desc', get_bloginfo('description'));

    ?>
    <div class="wrap">
        <h1>Fajaede SEO Instellingen</h1>

        <form method="post" action="options.php">
            <?php
            settings_fields('fajaede_seo_settings_group');
            do_settings_sections('fajaede_seo_settings_group');
            ?>
            <h2>Robots.txt Beheer</h2>
            <p>Pas hier je robots.txt aan. Wordt geserveerd zolang geen fysiek bestand aanwezig is.</p>
            <textarea name="fajaede_robots_txt" rows="15" style="width:100%; font-family: monospace;"><?php echo esc_textarea($robots_txt); ?></textarea>

            <h2>Social Media Handles</h2>
            <p>Voer Twitter en Facebook gebruikersnamen in (zonder @).</p>
            <table class="form-table" role="presentation">
                <tr>
                    <th><label for="fajaede_twitter_handle">Twitter Handle</label></th>
                    <td><input type="text" id="fajaede_twitter_handle" name="fajaede_twitter_handle" value="<?php echo esc_attr($twitter_handle); ?>" class="regular-text"></td>
                </tr>
                <tr>
                    <th><label for="fajaede_facebook_handle">Facebook Handle</label></th>
                    <td><input type="text" id="fajaede_facebook_handle" name="fajaede_facebook_handle" value="<?php echo esc_attr($facebook_handle); ?>" class="regular-text"></td>
                </tr>
            </table>

            <?php
            settings_fields('fajaedeseo_home_settings_group');
            ?>
            <h2>Homepagina SEO Instellingen</h2>
            <p>Specifieke SEO instellingen voor de homepage van je website.</p>
            <table class="form-table" role="presentation">
                <tr>
                    <th><label for="fajaedeseo_home_seo_title">Meta Title (Home)</label></th>
                    <td><input type="text" id="fajaedeseo_home_seo_title" name="fajaedeseo_home_seo_title" value="<?php echo esc_attr($home_title); ?>" class="regular-text"></td>
                </tr>
                <tr>
                    <th><label for="fajaedeseo_home_seo_desc">Meta Description (Home)</label></th>
                    <td><textarea id="fajaedeseo_home_seo_desc" name="fajaedeseo_home_seo_desc" rows="3" cols="60"><?php echo esc_textarea($home_desc); ?></textarea></td>
                </tr>
            </table>

            <?php submit_button('Opslaan'); ?>
            <p>Ga naar Instellingen &gt; Permalinks en klik op ‘Wijzigingen opslaan’ om rewrite rules te activeren.</p>
        </form>
    </div>
    <?php
}
add_action('admin_menu', function () {
    add_options_page(
        'Fajaede SEO',
        'Fajaede SEO',
        'manage_options',
        'fajaede-seo',
        'fajaedeseo_render_settings_page'
    );
});


// =============== Sitemap rewrite rule en output =================
add_filter('query_vars', fn($vars) => array_merge($vars, ['fajaede_sitemap']));
add_action('init', function () {
    add_rewrite_rule('^fajaede_sitemap/?$', 'index.php?fajaede_sitemap=1', 'top');
});
add_action('template_redirect', function () {
    if (!intval(get_query_var('fajaede_sitemap'))) return;

    header('Content-Type: application/xml; charset=UTF-8');
    header('Cache-Control: no-cache, must-revalidate');
    echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    echo '<?xml-stylesheet type="text/xsl" href="' . esc_url(plugins_url('sitemap-style.xsl', __FILE__)) . '"?>' . "\n";
    echo '<urlset xmlns="https://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

    $query = new WP_Query([
        'post_type' => ['post', 'page'],
        'post_status' => 'publish',
        'posts_per_page' => -1,
        'no_found_rows' => true,
        'update_post_meta_cache' => false,
        'update_post_term_cache' => false,
    ]);
    while ($query->have_posts()) {
        $query->the_post();
        printf("<url>\n\t<loc>%s</loc>\n\t<lastmod>%s</lastmod>\n\t<changefreq>weekly</changefreq>\n\t<priority>0.6</priority>\n</url>\n",
            esc_url(get_permalink()),
            esc_html(get_the_modified_date('c'))
        );
    }
    wp_reset_postdata();
    echo '</urlset>';
    exit;
});

// ================= Robots.txt dynamisch output =================
add_action('do_robots', function () {
    $robots_content = get_option('fajaede_robots_txt');
    if ($robots_content) {
        header('Content-Type: text/plain; charset=UTF-8');
       echo esc_textarea($robots_content);
        exit;
    }
});

// ================= SEO meta tags + canonical link =================
function fajaedeseo_add_meta_tags() {
    // Homepagina meta
    if ( (is_front_page() || is_home()) && !is_paged() ) {
        $meta_title = get_option('fajaedeseo_home_seo_title', get_bloginfo('name'));
        $meta_desc  = get_option('fajaedeseo_home_seo_desc', get_bloginfo('description'));
        if (!empty($meta_title)) {
            echo '<title>' . esc_html($meta_title) . "</title>\n";
        }
        if (!empty($meta_desc)) {
            echo '<meta name="description" content="' . esc_attr($meta_desc) . '">' . "\n";
        }
        // Canonical home URL
        echo '<link rel="canonical" href="' . esc_url(home_url('/')) . '">' . "\n";

        // Open Graph & Twitter Tags voor home
        $site_name = get_bloginfo('name');
        $locale = get_locale();
        $image_url = FAJAEDESEO_OG_IMAGE_FALLBACK;
        $site_icon_url = get_site_icon_url() ?: $image_url;
        $twitter_handle = get_option('fajaede_twitter_handle');
        $facebook_handle = get_option('fajaede_facebook_handle');

        echo '<meta property="og:type" content="website">' . "\n";
        echo '<meta property="og:title" content="' . esc_attr($meta_title) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr($meta_desc) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url(home_url('/')) . '">' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr($site_name) . '">' . "\n";
        echo '<meta property="og:image" content="' . esc_url($image_url) . '">' . "\n";
        echo '<meta property="og:locale" content="' . esc_attr($locale) . '">' . "\n";

        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr($meta_title) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr($meta_desc) . '">' . "\n";
        echo '<meta name="twitter:image" content="' . esc_url($image_url) . '">' . "\n";
        if ($twitter_handle) {
            echo '<meta name="twitter:site" content="@' . esc_attr($twitter_handle) . '">' . "\n";
            echo '<meta name="twitter:creator" content="@' . esc_attr($twitter_handle) . '">' . "\n";
        }
        if ($facebook_handle) {
            echo '<meta property="article:publisher" content="https://www.facebook.com/' . esc_attr($facebook_handle) . '"/>' . "\n";
        }
        // JSON-LD website voor home
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'WebSite',
            'url' => home_url('/'),
            'name' => $site_name,
            'publisher' => [
                '@type' => 'Organization',
                'name' => $site_name,
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => $site_icon_url,
                ],
            ],
        ];
        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
        return;
    }

    // Voor posts en pages
    if (!is_singular(['post', 'page'])) return;

    global $post;
    $meta_title = get_post_meta($post->ID, '_fajaedeseo_meta_title', true) ?: get_the_title($post);
    $meta_desc = get_post_meta($post->ID, '_fajaedeseo_meta_desc', true);

    if (empty($meta_desc)) {
        $meta_desc = wp_strip_all_tags(get_post_field('post_excerpt', $post) ?: get_post_field('post_content', $post));
        $meta_desc = mb_substr($meta_desc, 0, 160);
    }

    $url = get_permalink($post);
    $canonical = get_post_meta($post->ID, '_fajaedeseo_canonical', true) ?: $url;

    $site_name = get_bloginfo('name');
    $locale = get_locale();
    $modified_time = get_post_modified_time('c', false, $post);
    $author_name = get_the_author_meta('display_name', $post->post_author);
    $image_url = has_post_thumbnail($post) ? get_the_post_thumbnail_url($post, 'full') : FAJAEDESEO_OG_IMAGE_FALLBACK;
    $site_icon_url = get_site_icon_url() ?: $image_url;

    $twitter_handle = get_option('fajaede_twitter_handle');
    $facebook_handle = get_option('fajaede_facebook_handle');

    echo '<title>' . esc_html($meta_title) . "</title>\n";
    echo '<meta name="title" content="' . esc_attr($meta_title) . '">' . "\n";
    echo '<meta name="description" content="' . esc_attr($meta_desc) . '">' . "\n";

    echo '<link rel="canonical" href="' . esc_url($canonical) . '">' . "\n";

    // Open Graph
    echo '<meta property="og:type" content="article">' . "\n";
    echo '<meta property="og:title" content="' . esc_attr($meta_title) . '">' . "\n";
    echo '<meta property="og:description" content="' . esc_attr($meta_desc) . '">' . "\n";
    echo '<meta property="og:url" content="' . esc_url($url) . '">' . "\n";
    echo '<meta property="og:site_name" content="' . esc_attr($site_name) . '">' . "\n";
    echo '<meta property="og:image" content="' . esc_url($image_url) . '">' . "\n";
    echo '<meta property="og:locale" content="' . esc_attr($locale) . '">' . "\n";
    echo '<meta property="og:updated_time" content="' . esc_attr($modified_time) . '">' . "\n";

    // Twitter cards
    echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
    echo '<meta name="twitter:title" content="' . esc_attr($meta_title) . '">' . "\n";
    echo '<meta name="twitter:description" content="' . esc_attr($meta_desc) . '">' . "\n";
    echo '<meta name="twitter:image" content="' . esc_url($image_url) . '">' . "\n";
    if ($twitter_handle) {
        echo '<meta name="twitter:site" content="@' . esc_attr($twitter_handle) . '">' . "\n";
        echo '<meta name="twitter:creator" content="@' . esc_attr($twitter_handle) . '">' . "\n";
    }
    if ($facebook_handle) {
        echo '<meta property="article:publisher" content="https://www.facebook.com/' . esc_attr($facebook_handle) . '"/>' . "\n";
    }

    // JSON-LD structured data (Article)
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'Article',
        'mainEntityOfPage' => [
            '@type' => 'WebPage',
            '@id' => $url,
        ],
        'headline' => $meta_title,
        'description' => $meta_desc,
        'image' => [
            '@type' => 'ImageObject',
            'url' => $image_url,
            'width' => 1200,
            'height' => 630,
        ],
        'author' => [
            '@type' => 'Person',
            'name' => $author_name,
        ],
        'publisher' => [
            '@type' => 'Organization',
            'name' => $site_name,
            'logo' => [
                '@type' => 'ImageObject',
                'url' => $site_icon_url,
                'width' => 512,
                'height' => 512,
            ],
        ],
        'datePublished' => get_the_date(DATE_W3C, $post),
        'dateModified' => get_the_modified_date(DATE_W3C, $post),
        'url' => $url,
    ];
    echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
}
add_action('wp_head', 'fajaedeseo_add_meta_tags');


// ================= Robots.txt dynamisch output =================
add_action('do_robots', function () {
    $robots_content = get_option('fajaede_robots_txt');
    if ($robots_content) {
        header('Content-Type: text/plain; charset=UTF-8');
        echo esc_textarea($robots_content);
        exit;
    }
});

// ================= Gutenberg Sidebar met SEO info & Site Audit =================
function fajaedeseo_register_sidebar_script() {
    wp_register_script(
        'fajaedeseo-sidebar',
        plugins_url('fajaedeseo-sidebar.js', __FILE__),
        ['wp-api-fetch', 'wp-element', 'wp-components', 'wp-data', 'wp-edit-post', 'wp-plugins'],
        filemtime(plugin_dir_path(__FILE__) . 'fajaedeseo-sidebar.js'),
        true
    );
    wp_enqueue_script('fajaedeseo-sidebar');
}
add_action('enqueue_block_editor_assets', 'fajaedeseo_register_sidebar_script');


// Voorbeeld van REST endpoint voor site ranking & audit data (dummy data)
add_action('rest_api_init', function () {
    register_rest_route('fajaedeseo/v1', '/site-audit', [
        'methods' => 'GET',
        'callback' => function () {
            // Simulatie van een site audit en ranking score
            return [
                'overall_score' => 78,
                'seo_issues' => [
                    ['issue' => 'Ontbrekende alt tags', 'count' => 4],
                    ['issue' => 'Langzame paginasnelheid', 'count' => 2],
                    ['issue' => 'Geen XML sitemap gevonden', 'count' => 0]
                ],
                'recommendations' => [
                    'Gebruik alt tags bij afbeeldingen',
                    'Optimaliseer afbeeldingen en cache',
                    'Controleer sitemap instelling'
                ],
            ];
        },
        'permission_callback' => fn() => current_user_can('edit_posts')
    ]);
});
